### A Pluto.jl notebook ###
# v0.20.4

using Markdown
using InteractiveUtils

# ╔═╡ 05ad5ab7-70f8-4bdb-8bbf-e05822cc5623
# ╠═╡ show_logs = false
import Pkg; Pkg.activate(".")

# ╔═╡ d2361176-db34-450c-8721-ed4391b90e9c
begin
	using CairoMakie
	using CommonMark
	using ContinuousWavelets
	using Distributions
	using DSP
	using PlutoUI
	using Random
end

# ╔═╡ 0118270a-090b-4237-9002-bd68b608a307
md"""
**What is this?**


*This jupyter notebook is part of a collection of notebooks on various topics discussed during the Time Domain Astrophysics course delivered by Stefano Covino at the [Università dell'Insubria](https://www.uninsubria.eu/) in Como (Italy). Please direct questions and suggestions to [stefano.covino@inaf.it](mailto:stefano.covino@inaf.it).*
"""

# ╔═╡ de92ed35-d7b6-40f9-a642-d9a819804c85
md"""
**This is a `Julia` notebook**
"""

# ╔═╡ 1f00bd93-c681-4b4f-ba15-f4f67b3b40e3
Pkg.instantiate()

# ╔═╡ 5049cda4-f875-4291-a7e5-59a09b57e4e8
# ╠═╡ show_logs = false
Pkg.instantiate()

# ╔═╡ f40ad72a-b6fd-4024-ab09-de1b6e31501b
md"""
$(LocalResource("Pics/TimeDomainBanner.jpg"))
"""

# ╔═╡ 090ffe2f-b381-49ec-bc08-53219c690e77
md"""
# Wavelet Analysis

***

## PSD Varying Time-Series
***

- The trigonometric basis functions used in the Fourier transform have an infinite extent and for this reason the Fourier transform may not be the best method to analyze non periodic time series data, such as the case of a localized event (e.g., a burst that decays over some time scale so that the PSD is also varying with time).

- We can (and actually do) evaluate the PSD for finite stretches of time series in order to detect its changes.

- This approach (called *spectrogram*, or *dynamical power spectra analysis*, or even *windowed Fourier transform*) suffers from degraded spectral resolution and is sensitive to the specific choice of time series segmentation length.

### Windowed Fourier transform
***

- The FT is performed on a sliding segment of length $T$ from a time series of time step $δt$ and total length $Nδt$, thus returning frequencies from $T^{−1}$ to $2δt^{−1}$ at each time step. 

- The segments can be windowed with an arbitrary function such as a boxcar (no smoothing) or any other choice.

- The WFT represents an inaccurate and inefficient method of time–frequency localization, as it imposes a scale or “response interval” T into the analysis. 

- The inaccuracy arises from the aliasing of high- and low-frequency components that do not fall within the frequency range of the window. The inefficiency comes from the $T/(2δt)$ frequencies, which must be analyzed at each time step.


"""

# ╔═╡ 6721746e-71e6-4d3d-a32a-6c0385986351
md"""
## Wavelets
***

- A popular family of basis functions is called wavelets.

    - By construction, wavelets are localized in both frequency and time domains. 

- Individual wavelets are specified by a set of *wavelet filter coefficients*. Given a wavelet, a complete orthonormal set of basis functions can be constructed by scalings and translations.

- Different wavelet families trade the localization of a wavelet with its smoothness. Popular wavelets include “Mexican hat”, Haar and Daubechies wavelets. 

- A wavelet is a “small wave,” where small derives from the fact that it is mostly limited to an interval of time. 

- A (mother) wavelet $ψ$ has to satisfy two requirements: its integral must be zero and the integral of its square must be unity:

```math
\int_{-\infty}^\infty \Psi(t) dt = 0 \qquad \int_{-\infty}^\infty \Psi^2(t) dt = 1
```

- It is easy to see that these requirements imply that $ψ(t)$ is essentially non-zero only over a limited range of t and that it has to extend both above and below zero.

$(LocalResource("Pics/wavelets.png"))

- Three examples of wavelets. From left to right: Haar wavelet, mexican hat and morlet wavelet.
"""

# ╔═╡ 0bac9591-9018-47af-b567-ec756f8c6c42
cm"""
- A wavelet can be shifted by ``τ`` and dilated by a scale parameter ``σ``:

```math
 \Psi_{\tau,\sigma}(t) = \frac{1}{\sqrt{\sigma}} \Psi\left(\frac{t-\tau}{\sigma}\right) 
```

- Again ``ψ(t)`` is essentially non-zero only over a limited range of ``t`` and that it has to extend both above and below zero.


    
- The wavelet transform of a function ``f(t)`` is computed by correlating ``f(t)`` with the complex conjugate of ``ψ_{τ,σ}(t)`` (wavelets can also be complex functions):

```math
W[f(\tau,\sigma)] = \int_{-\infty}^\infty f(t) \Psi_{\tau,\sigma}^*(t)dt
```
    
- The *discrete wavelet transform* (DWT) can be used to analyze the power spectrum of a time series as a function of time.

- A possible wavelet might be: ``w(t|t_0, f_0, Q) = A \exp[i 2 \pi f_0 (t-t_0)] \exp[-f_0^2 (t-t_0)^2 / Q^2]``,
    - where ``t_0``  is the central time, ``f_0``  is the central frequency, and the dimensionless parameter ``Q`` is a model parameter which controls the width of the frequency window.

$(LocalResource("Pics/waveletex1.png"))
    
- The Fourier transform of the previous wavelet is:

```math
W(t|t_0, f_0, Q) = \left( \frac{\pi}{f_0^2/Q^2} \right)^{1/2} \exp[i 2 \pi f_0 t_0] \exp[-\frac{\pi^2 Q^2 (f-f_0)^2}{Q f_0^2}]
```

- Technically speaking, this is not exactly a wavelet. It is indeed closely related to a true wavelet, the Morlet wavelet, through a simple scaling and offset.
    - One might think to it as a “matched filters”.

- Given a signal ``h(t)``, its wavelet transform is given by the convolution:

```math
H_w(t_0; f_0, Q) = \int_{-\infty}^\infty h(t)w(t|t_0, f_0, Q)
```
    
- By the convolution theorem, we can write the Fourier transform of ``H_w``  as the pointwise product of the Fourier transforms of ``h(t)`` (e.g. by a DFT) and of ``w(t | t_0, f_0, Q)``.
    
- Then, the wavelet PSD, can be defined by:

```math
     {\rm PSD}_w(f_0, t_0; Q) = |H_w(t_0; f_0, Q)|^2
```
    
- As said, unlike the typical Fourier-transform PSD, the wavelet PSD allows detection of frequency information which is localized in time.

- This is indeed one the approaches (we are simplifying a lot here…) used in the, e.g., LIGO/Virgo projects to detect gravitational wave events. 

- Because of the noise level in the GW measurements, rather than a standard wavelet they instead use functions which are tuned to the expected form of the signal (i.e., matched filters).


        
- Example with a localized Gaussian noise:
    
$(LocalResource("Pics/waveletex2.png"))

- The upper panel shows the signal. The middle panel shows an example wavelet and the lower panel shows the power spectral density as a function of the frequency ``f_0`` and the time ``t_0``, for ``Q = 1.0``.



- And now a different example:

(LocalResource("Pics/waveletex3.png"))

- The upper panel shows the input signal, which consists of a Gaussian spike in the presence of white (Gaussian) noise. The middle panel shows again an example wavelet. The lower panel shows the PSD as a function of the frequency ``f_0`` and the time ``t_0``, for ``Q = 0.3``.
"""

# ╔═╡ a37c5419-797a-4626-8138-2f2d5ab8f416
md"""
#### Exercize: wavelet analysis of a chirp signal
***

- Let's define a "chirp" signal:
"""

# ╔═╡ f3f93bff-1c39-4652-aa41-0b7d88ad9caf
begin
	function chirp(t, T, A, phi, omega, beta)
	    signal = A .* sin.(phi .+ omega .* (t .- T) .+ beta .* (t .- T).^2)
	    signal[t .< T] .= 0
	    return signal
	end
	
	N = 4096
	t = range(-50,50,N)
	h_true = chirp(t, -20, 0.8, 0, 0.2, 0.02)
	d = Normal(0.,1.)
	h = h_true .+ rand(d,N)
end;

# ╔═╡ 13fbe8ec-5266-4b72-b85e-f171c8c7ba90
begin
	fg1 = Figure()
	
	ax1fg1 = Axis(fg1[1, 1],
	    xlabel="Time (s)",
	    ylabel="Flux (arbitrary units)",
	    )
	
	lines!(t,h,label="Chirp signal")
	lines!(t,h_true,label="Chirp signal without noise")
	
	axislegend()
	
	fg1
end

# ╔═╡ 8550f405-2819-4a33-9cd2-c72dbbb04a87
md"""
- Let's now compute a Fourier periodogram
"""

# ╔═╡ 0e3b9aaf-0d91-4c60-8118-5624273d3983
begin
	# We use the DSP package
	
	dt = 100/4096
	psd = periodogram(h, fs=1/dt)
	
	fg2 = Figure()
	
	ax1fg2 = Axis(fg2[1, 1],
	    xlabel = "Frequency (Hz)",
	    ylabel = "Power",
	    )
	
	lines!(psd.freq,psd.power,color=:blue,label="PSD")
	
	xlims!(0,1)
	
	axislegend()
	
	
	fg2
end

# ╔═╡ 7b3e4dfa-8dcd-4b5c-bc05-e1d51449d8c3
md"""
- The periodogram shows power at the lowest frequencies with a break at frequencies higher than $\sim 0.5$Hz, i.e. for time-scales shorter than $\sim 1$s.

- Let's try to study the time-evolution of the PSD by a *spectrogram*:
"""

# ╔═╡ 738b344d-e2ae-4f99-b755-38c0dcdc3da3
begin
	tper = spectrogram(h,div(length(h),5),0,fs=1/dt)
	
	fg3 = Figure()
	
	axfg3 = Axis(fg3[1, 1],
	    xlabel="Time (s)",
	    ylabel="Frequency (Hz)",
	    )
	
	p = heatmap!(tper.time.-50,tper.freq,tper.power',colorscale=log10)
	Colorbar(fg3[:, end+1], p)
	
	ylims!(0,1)
	
	fg3
	
end

# ╔═╡ 576eb9f0-40c6-4a9a-9504-d562452541e6
md"""
- And, it is indeed clearly visible a signal appearing at about -20s with frequncy increasing with time.

- Nevertheless, this is not an optimal solution since the number of bins is arbitrary, and clearly we reduce the resolution of the analysis decreasing the length of the time-series.



- This is where a wavelet analysis can give a contribution.
"""

# ╔═╡ 6e81c233-8aa5-4ab4-a011-c56ad0026ea9
begin
	c = wavelet(Morlet(π), Q=8, β=1, averagingType=NoAve(), p=4)
	
	res = cwt(h, c)
end;

# ╔═╡ 6a71ccd0-8ee7-4c39-b8c7-88b398e98d1a
begin
	fg4 = Figure()
	
	axfg4 = Axis(fg4[1, 1],
	    xlabel=L"Time index $\rightarrow$",
	    ylabel=L"Frequency index $\rightarrow$",
	    xticklabelsvisible=false,
	    yticklabelsvisible=false,
	    )
	
	p4 = heatmap!(abs.(res))
	Colorbar(fg4[:, end+1], p4)
	
	fg4
end

# ╔═╡ f153fdfa-57d5-4112-8d59-847ee1ec06ca
md"""
- And we can appreciate how the formation and evolution of the signal is clealry visible in the plot.
"""

# ╔═╡ bde447ae-3998-4a1a-8e4e-32f97b2854a1
md"""
## Reference & Material

Material and papers related to the topics discussed in this lecture.

- [Belloni & Bhattacharya (2022) - "Basics of Fourier Analysis for High-Energy Astronomy”](https://ui.adsabs.harvard.edu/abs/2022hxga.book....7B/abstract)
- [Ivezić et al. (2020) - "Statistics, Data Mining, and Machine Learning in Astronomy"](https://ui.adsabs.harvard.edu/abs/2020sdmm.book.....I/abstract)
"""

# ╔═╡ c4b73630-ba8d-4403-9b7b-b498eac9792b
md"""
## Further Material

Papers for examining more closely some of the discussed topics.

- [Torrence & Compo (1998) - "A Practical Guide to Wavelet Analysis"](https://ui.adsabs.harvard.edu/abs/1998BAMS...79...61T/abstract)


"""

# ╔═╡ af38d725-6e0f-4958-85bb-1be029ba3fb5
md"""
### Credits
***

This notebook contains material obtained from [https://www.astroml.org/book_figures/chapter10/fig_chirp2_PSD.html](https://www.astroml.org/book_figures/chapter10/fig_chirp2_PSD.html).
"""

# ╔═╡ 241907cb-1fb0-4963-8bb9-7894136e43de
cm"""
## Course Flow

<table>
  <tr>
    <td>Previous lecture</td>
    <td>Next lecture</td>
  </tr>
  <tr>
    <td><a href="./open?path=Lectures/Science Case - AGN and Blazars/Lecture-AGN-and-Blazars.jl">Science case about AGN and blazars</a></td>
    <td><a href="./open?path=Lecture - Time of Arrival/Lecture-Time-of-Arrival.jl">Lecture about time of arrival analysis</a></td>
  </tr>
 </table>


"""

# ╔═╡ d3b81af7-b85b-4212-82a2-0f97ec31a484
md"""
**Copyright**

This notebook is provided as [Open Educational Resource](https://en.wikipedia.org/wiki/Open_educational_resources). Feel free to use the notebook for your own purposes. The text is licensed under [Creative Commons Attribution 4.0](https://creativecommons.org/licenses/by/4.0/), the code of the examples, unless obtained from other properly quoted sources, under the [MIT license](https://opensource.org/licenses/MIT). Please attribute the work as follows: *Stefano Covino, Time Domain Astrophysics - Lecture notes featuring computational examples, 2025*.
"""

# ╔═╡ Cell order:
# ╟─0118270a-090b-4237-9002-bd68b608a307
# ╟─de92ed35-d7b6-40f9-a642-d9a819804c85
# ╟─05ad5ab7-70f8-4bdb-8bbf-e05822cc5623
# ╟─1f00bd93-c681-4b4f-ba15-f4f67b3b40e3
# ╟─5049cda4-f875-4291-a7e5-59a09b57e4e8
# ╟─d2361176-db34-450c-8721-ed4391b90e9c
# ╟─f40ad72a-b6fd-4024-ab09-de1b6e31501b
# ╟─090ffe2f-b381-49ec-bc08-53219c690e77
# ╟─6721746e-71e6-4d3d-a32a-6c0385986351
# ╟─0bac9591-9018-47af-b567-ec756f8c6c42
# ╟─a37c5419-797a-4626-8138-2f2d5ab8f416
# ╟─f3f93bff-1c39-4652-aa41-0b7d88ad9caf
# ╟─13fbe8ec-5266-4b72-b85e-f171c8c7ba90
# ╟─8550f405-2819-4a33-9cd2-c72dbbb04a87
# ╟─0e3b9aaf-0d91-4c60-8118-5624273d3983
# ╟─7b3e4dfa-8dcd-4b5c-bc05-e1d51449d8c3
# ╟─738b344d-e2ae-4f99-b755-38c0dcdc3da3
# ╟─576eb9f0-40c6-4a9a-9504-d562452541e6
# ╟─6e81c233-8aa5-4ab4-a011-c56ad0026ea9
# ╟─6a71ccd0-8ee7-4c39-b8c7-88b398e98d1a
# ╟─f153fdfa-57d5-4112-8d59-847ee1ec06ca
# ╟─bde447ae-3998-4a1a-8e4e-32f97b2854a1
# ╟─c4b73630-ba8d-4403-9b7b-b498eac9792b
# ╟─af38d725-6e0f-4958-85bb-1be029ba3fb5
# ╟─241907cb-1fb0-4963-8bb9-7894136e43de
# ╟─d3b81af7-b85b-4212-82a2-0f97ec31a484
